package com.agilex.healthcare.directscheduling.mdws.translator;


import java.util.Date;
import java.util.Iterator;

import com.agilex.healthcare.directscheduling.mdws.generatedwsdl.schedulingservice.AppointmentTO;
import com.agilex.healthcare.directscheduling.mdws.generatedwsdl.schedulingservice.HospitalLocationTO;
import com.agilex.healthcare.directscheduling.mdws.generatedwsdl.schedulingservice.TaggedAppointmentArray;
import com.agilex.healthcare.directscheduling.mdws.generatedwsdl.schedulingservice.TaggedText;
import com.agilex.healthcare.directscheduling.validator.MdwsResponseException;
import com.agilex.healthcare.directscheduling.domain.BookedAppointment;
import com.agilex.healthcare.directscheduling.domain.BookedAppointments;
import com.agilex.healthcare.directscheduling.domain.SchedulingClinic;
import com.agilex.healthcare.directscheduling.utils.DateHelper;
import com.agilex.healthcare.utility.NullChecker;

public class BookedAppointmentTranslator {

	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(BookedAppointmentTranslator.class);

    public BookedAppointments translateFromTaggedAppointmentArray(TaggedAppointmentArray arr) {
    	BookedAppointments retValue = new BookedAppointments();
    	if (arr == null)
    		throw new MdwsResponseException(MdwsResponseException.UserMessages.INVALID_DATA_RETURNED, "TaggedAppointmentArray was null");
    	if (arr.getFault() != null)
    		throw new MdwsResponseException(arr.getFault());
    	if (arr.getAppts() == null || arr.getAppts().getAppointmentTO() == null)
    		return retValue;

    	Iterator<AppointmentTO> itAppointmentTO = arr.getAppts().getAppointmentTO().iterator();
    	while (itAppointmentTO.hasNext()) {
    		AppointmentTO appointmentTO = itAppointmentTO.next();
    		//Perform Error Checking
    		if (appointmentTO.getFault() != null)
    			throw new MdwsResponseException(appointmentTO.getFault());

    		BookedAppointment bookedAppointment = createBookedAppointment(appointmentTO);
    		retValue.add(bookedAppointment);
    	}

    	return retValue;
    }

    private BookedAppointment createBookedAppointment(AppointmentTO appointmentTO) {
    	String appointmentLength = appointmentTO.getAppointmentLength();
		String timestamp = appointmentTO.getTimestamp();
		Date appointmentDate = DateHelper.parseVistaDateTime(timestamp);
		String labDateTime = appointmentTO.getLabDateTime();
		String xrayDateTime = appointmentTO.getXrayDateTime();
		String ekgDateTime = appointmentTO.getEkgDateTime();
		String purpose = appointmentTO.getPurpose();
		String type = appointmentTO.getType();
		String currentStatus = appointmentTO.getCurrentStatus();
		String bookNotes = appointmentTO.getBookNotes();

		BookedAppointment bookedAppointment = new BookedAppointment();

        bookedAppointment.setAppointmentLength(appointmentLength);
        bookedAppointment.setAppointmentTime(appointmentDate);

        setClinic(bookedAppointment, appointmentTO);

        if (NullChecker.isNotNullish(labDateTime)) {
			Date parseVistaDateTime = parseOtherProcedureDates(labDateTime);
			bookedAppointment.setLabDateTime(parseVistaDateTime);
		}
        if (NullChecker.isNotNullish(xrayDateTime)) {
			Date parseVistaDateTime = parseOtherProcedureDates(xrayDateTime);
			bookedAppointment.setXrayDateTime(parseVistaDateTime);
		}
        if (NullChecker.isNotNullish(ekgDateTime)) {
			Date parseVistaDateTime = parseOtherProcedureDates(ekgDateTime);
			bookedAppointment.setEkgDateTime(parseVistaDateTime);
		}
        bookedAppointment.setPurpose(purpose);
        bookedAppointment.setType(type);
        bookedAppointment.setCurrentStatus(currentStatus);
        bookedAppointment.setBookingNote(bookNotes);

        return bookedAppointment;
    }

    private void setClinic(BookedAppointment bookedAppointment, AppointmentTO appointmentTO) {
    	SchedulingClinic clinic = new SchedulingClinic();
    	String clinicID = appointmentTO.getClinicID();
		clinic.setId(clinicID);

    	HospitalLocationTO hospitalLocationTO = appointmentTO.getClinic();
    	if (hospitalLocationTO != null) {
    		//Perform Error Checking
    		if (hospitalLocationTO.getFault() != null)
    			throw new MdwsResponseException(hospitalLocationTO.getFault());

    		TaggedText departmentTag = hospitalLocationTO.getDepartment();
    		TaggedText serviceTag = hospitalLocationTO.getService();
    		TaggedText specialtyTag = hospitalLocationTO.getSpecialty();
    		TaggedText stopCodeTag = hospitalLocationTO.getStopCode();
    		//Perform Error Checking
    		if (departmentTag.getFault() != null)
    			throw new MdwsResponseException(departmentTag.getFault());
    		//Perform Error Checking
    		if (serviceTag.getFault() != null)
    			throw new MdwsResponseException(serviceTag.getFault());
    		//Perform Error Checking
    		if (specialtyTag.getFault() != null)
    			throw new MdwsResponseException(specialtyTag.getFault());
    		//Perform Error Checking
    		if (stopCodeTag.getFault() != null)
    			throw new MdwsResponseException(stopCodeTag.getFault());

    		String name = hospitalLocationTO.getName();
			String department = departmentTag.getText();
			String service = serviceTag.getText();
			String specialty = specialtyTag.getText();
			String stopCode = stopCodeTag.getText();
			boolean askForCheckIn = hospitalLocationTO.isAskForCheckIn();

	        clinic.setName(name);
	        clinic.setDepartment(department);
	        clinic.setService(service);
	        clinic.setSpecialty(specialty);
	        clinic.setStopCode(stopCode);
	        clinic.setAskForCheckIn(askForCheckIn);
//	        clinic.setFacility(extractFacilityInfo(hospitalLocationTO.getFacility()));
    	}
        bookedAppointment.setClinic(clinic);
    }

	private Date parseOtherProcedureDates (String date) {
		Date parsedDate;
		try {
			parsedDate = DateHelper.parseVistaDateTime(date);
			return parsedDate;
		} catch (Exception e) {
			logger.warn("parsing date exception", e);
			return null;
		}
	}
}
